// vim: set ts=4 :
///////////////////////////////////////////////////////////
//  rbr.plugin.IService.h
//  Implementation of the Class IService
//  Created on:      09-Jan-2021 09:19:35
//  Original author: Guenter Schlupf
///////////////////////////////////////////////////////////
#pragma once


namespace rbr { namespace plugin { 

/// <summary>
/// The IService interface provides methods to handle car damage and service park.
/// 
/// </summary>
class IService
{

public:
    /// <summary>
    /// Initializes the service with the given car and service park data.
    /// </summary>
    /// <param name="car">The car index.</param>
    /// <param name="numMechanics">The number of mechanics working in the service. If
    /// zero, no repair takes place, up to 6 allowed!</param>
    /// <param name="mechanicsSkill">The skill of the mechanics working on the car.
    /// </param>
    /// <param name="allowedRepairTime">The allowed repair time in minutes.</param>
    virtual void init(
            int car,
            int numMechanics,
            int mechanicsSkill,
            float allowedRepairTime) = 0;
    /// <summary>
    /// Releases allocated resources after servicing the car in the repair car menu
    /// page. Needs to be called when you are done with servicing the car!
    /// </summary>
    virtual void finish() = 0;
    /// <summary>
    /// Loads the repair car page.
    /// </summary>
    virtual void repairCar() = 0;
    /// <summary>
    /// Sets the car damage handler as byte array. Size is 0x3EB0.
    /// Can be called after init and before repairCar to set the players current car
    /// damage.
    /// </summary>
    /// <returns>The number of bytes copied. Max. 0x3EB0.</returns>
    /// <param name="pBuffer">The source buffer with appropriate size from which the
    /// damage data is to be copied into RBR's car damage handler.</param>
    /// <param name="bufSize">The size of the provided buffer in bytes.</param>
    virtual size_t setDamageBuffer(const char* pBuffer, size_t bufSize)
            = 0;
    /// <summary>
    /// Returns the car damage handler as byte array. Size is 0x3EB0.
    /// Should be called after finish(), as then the repaired state is reflected in
    /// the internal damage handler.
    /// </summary>
    /// <returns>The number of bytes copied. Max. 0x3EB0.</returns>
    /// <param name="pBuffer">The target buffer with appropriate size into which the
    /// damage data is to be  copied.</param>
    /// <param name="bufSize">The size of the provided buffer in bytes.</param>
    virtual size_t getDamageBuffer(char* pBuffer, size_t bufSize) = 0;
    /// <summary>
    /// Returns the penalty in seconds, applied when exceeding the allowed repair
    /// time.
    /// </summary>
    virtual float getPenalty() = 0;
    /// <summary>
    /// Resets the damage of the car.
    /// </summary>
    virtual void resetDamage() = 0;
    /// <summary>
    /// Returns whether the car is in a drivable state. If not, it is beyond repair.
    /// </summary>
    virtual int isCarDrivable() = 0;

};

}}



